<?php
/**
 * PHP Live! PHP 8.2 Upgrade Script
 * This script replaces deprecated [) constructs with modern array destructuring
 */

echo "Starting PHP 8.2 upgrade...\n";

// Function to recursively find and replace [) constructs in PHP files
function upgradeListConstructs($directory) {
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    $phpFiles = new RegexIterator($iterator, '/\.php$/');
    
    foreach ($phpFiles as $file) {
        $filePath = $file->getPathname();
        $content = file_get_contents($filePath);
        
        // Skip if no [) constructs found
        if (strpos($content, '[') === false) {
            continue;
        }
        
        echo "Processing: " . $filePath . "\n";
        
        // Replace [ with [
        $content = preg_replace('/LIST\s*\(/', '[', $content);
        
        // Replace ) with ]
        $content = preg_replace('/\)\s*=/', '] =', $content);
        
        // Write back the updated content
        file_put_contents($filePath, $content);
        
        echo "  - Updated [) constructs\n";
    }
}

// Function to update PHP version requirements in setup files
function updatePHPVersionChecks($directory) {
    $files = [
        'setup/install.php',
        'setup/interface.php',
        'setup/interface_lang.php',
        'setup/interface_themes.php',
        'setup/interface_color.php',
        'API/Util_Error.php',
        'API/Util_Vals.php'
    ];
    
    foreach ($files as $file) {
        $filePath = $directory . '/' . $file;
        if (file_exists($filePath)) {
            $content = file_get_contents($filePath);
            
            // Replace old PHP version checks
            $content = preg_replace('/if\s*\(\s*phpversion\(\)\s*>=\s*["\']5\.1\.0["\']\s*\)\s*\{/i', '', $content);
            $content = preg_replace('/\}\s*;\s*$/', ';', $content);
            
            // Add modern timezone setting
            $content = str_replace(
                'date_default_timezone_set( "America/New_York" ) ;',
                '// PHP 8.0+ compatibility - timezone is always available',
                $content
            );
            
            file_put_contents($filePath, $content);
            echo "Updated: " . $file . "\n";
        }
    }
}

// Function to create a modern .htaccess file
function createModernHtaccess($directory) {
    $htaccessContent = <<<'EOT'
# PHP Live! - Modern .htaccess for PHP 8.2+
# Enable rewrite engine
RewriteEngine On

# Force HTTPS (uncomment if using SSL)
# RewriteCond %{HTTPS} off
# RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]

# Security headers
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
</IfModule>

# PHP settings for PHP 8.2
<IfModule mod_php8.c>
    php_value max_execution_time 300
    php_value memory_limit 256M
    php_value post_max_size 64M
    php_value upload_max_filesize 64M
    php_value max_input_vars 3000
    php_value session.gc_maxlifetime 1440
    php_value session.cookie_lifetime 0
    php_flag session.use_strict_mode On
    php_flag session.use_cookies On
    php_flag session.use_only_cookies On
    php_flag session.cookie_httponly On
    php_flag session.cookie_samesite "Lax"
</IfModule>

# Fallback for older PHP versions
<IfModule mod_php7.c>
    php_value max_execution_time 300
    php_value memory_limit 256M
    php_value post_max_size 64M
    php_value upload_max_filesize 64M
    php_value max_input_vars 3000
    php_value session.gc_maxlifetime 1440
    php_value session.cookie_lifetime 0
    php_flag session.use_strict_mode On
    php_flag session.use_cookies On
    php_flag session.use_only_cookies On
    php_flag session.cookie_httponly On
</IfModule>

# Cache control for static assets
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 month"
    ExpiresByType image/jpeg "access plus 1 month"
    ExpiresByType image/gif "access plus 1 month"
    ExpiresByType image/png "access plus 1 month"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
    ExpiresByType text/javascript "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType application/x-javascript "access plus 1 month"
    ExpiresByType application/x-shockwave-flash "access plus 1 month"
    ExpiresByType image/x-icon "access plus 1 year"
    ExpiresDefault "access plus 2 days"
</IfModule>

# Gzip compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/plain
    AddOutputFilterByType DEFLATE text/html
    AddOutputFilterByType DEFLATE text/xml
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/xml
    AddOutputFilterByType DEFLATE application/xhtml+xml
    AddOutputFilterByType DEFLATE application/rss+xml
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE application/x-javascript
</IfModule>

# Prevent access to sensitive files
<FilesMatch "\.(htaccess|htpasswd|ini|log|sh|inc|bak)$">
    Order Allow,Deny
    Deny from all
</FilesMatch>

# Prevent access to config files
<FilesMatch "^(config|database|\.env)">
    Order Allow,Deny
    Deny from all
</FilesMatch>
EOT;
    
    file_put_contents($directory . '/.htaccess', $htaccessContent);
    echo "Created modern .htaccess file\n";
}

// Function to create a composer.json file for modern dependency management
function createComposerJson($directory) {
    $composerContent = <<<'EOT'
{
    "name": "osicodes/phplive",
    "description": "PHP Live! Chat Application - Upgraded for PHP 8.2+",
    "type": "application",
    "keywords": ["chat", "live-chat", "customer-support"],
    "license": "proprietary",
    "require": {
        "php": ">=8.0",
        "ext-mysqli": "*",
        "ext-json": "*",
        "ext-mbstring": "*",
        "ext-curl": "*"
    },
    "require-dev": {
        "phpunit/phpunit": "^9.0"
    },
    "autoload": {
        "psr-4": {
            "PHPLive\\": "src/"
        }
    },
    "config": {
        "optimize-autoloader": true,
        "sort-packages": true
    },
    "minimum-stability": "stable",
    "prefer-stable": true
}
EOT;
    
    file_put_contents($directory . '/composer.json', $composerContent);
    echo "Created composer.json file\n";
}

// Function to create a modern error handler
function createModernErrorHandler($directory) {
    $errorHandlerContent = <<<'EOT'
<?php
/**
 * Modern Error Handler for PHP 8.2+
 * Compatible with PHP Live! application
 */

if (!defined('MODERN_ERROR_HANDLER')) {
    define('MODERN_ERROR_HANDLER', true);
    
    // Set error reporting for development/production
    if (defined('PHPLIVE_DEBUG') && PHPLIVE_DEBUG) {
        error_reporting(E_ALL);
        ini_set('display_errors', 1);
    } else {
        error_reporting(E_ALL & ~E_DEPRECATED & ~E_STRICT);
        ini_set('display_errors', 0);
    }
    
    // Custom error handler
    function modernErrorHandler($errno, $errstr, $errfile, $errline) {
        if (!(error_reporting() & $errno)) {
            return false;
        }
        
        $errorType = match($errno) {
            E_ERROR => 'Fatal Error',
            E_WARNING => 'Warning',
            E_PARSE => 'Parse Error',
            E_NOTICE => 'Notice',
            E_CORE_ERROR => 'Core Error',
            E_CORE_WARNING => 'Core Warning',
            E_COMPILE_ERROR => 'Compile Error',
            E_COMPILE_WARNING => 'Compile Warning',
            E_USER_ERROR => 'User Error',
            E_USER_WARNING => 'User Warning',
            E_USER_NOTICE => 'User Notice',
            E_STRICT => 'Strict',
            E_RECOVERABLE_ERROR => 'Recoverable Error',
            E_DEPRECATED => 'Deprecated',
            E_USER_DEPRECATED => 'User Deprecated',
            default => 'Unknown Error'
        };
        
        $errorMessage = "[$errorType] $errstr in $errfile on line $errline";
        
        // Log error
        error_log($errorMessage);
        
        // For fatal errors, we might want to show a user-friendly message
        if (in_array($errno, [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            if (defined('PHPLIVE_DEBUG') && PHPLIVE_DEBUG) {
                echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; margin: 10px; border-radius: 4px;'>";
                echo "<strong>Fatal Error:</strong> $errstr<br>";
                echo "<strong>File:</strong> $errfile<br>";
                echo "<strong>Line:</strong> $errline";
                echo "</div>";
            } else {
                echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; margin: 10px; border-radius: 4px;'>";
                echo "<strong>System Error:</strong> An unexpected error occurred. Please try again later.";
                echo "</div>";
            }
        }
        
        return true;
    }
    
    // Set custom error handler
    set_error_handler('modernErrorHandler');
    
    // Handle fatal errors
    register_shutdown_function(function() {
        $error = error_get_last();
        if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            modernErrorHandler($error['type'], $error['message'], $error['file'], $error['line']);
        }
    });
}
EOT;
    
    file_put_contents($directory . '/API/Modern_Error_Handler.php', $errorHandlerContent);
    echo "Created modern error handler\n";
}

// Main upgrade process
try {
    $currentDir = __DIR__;
    
    echo "Current directory: $currentDir\n";
    
    // Step 1: Upgrade [) constructs
    echo "\nStep 1: Upgrading [) constructs...\n";
    upgradeListConstructs($currentDir);
    
    // Step 2: Update PHP version checks
    echo "\nStep 2: Updating PHP version checks...\n";
    updatePHPVersionChecks($currentDir);
    
    // Step 3: Create modern .htaccess
    echo "\nStep 3: Creating modern .htaccess...\n";
    createModernHtaccess($currentDir);
    
    // Step 4: Create composer.json
    echo "\nStep 4: Creating composer.json...\n";
    createComposerJson($currentDir);
    
    // Step 5: Create modern error handler
    echo "\nStep 5: Creating modern error handler...\n";
    createModernErrorHandler($currentDir);
    
    echo "\n✅ PHP 8.2 upgrade completed successfully!\n";
    echo "\nNext steps:\n";
    echo "1. Test the application with PHP 8.2\n";
    echo "2. Run: composer install (if you want to use Composer)\n";
    echo "3. Update your web server configuration\n";
    echo "4. Test all functionality thoroughly\n";
    
} catch (Exception $e) {
    echo "\n❌ Error during upgrade: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . "\n";
    echo "Line: " . $e->getLine() . "\n";
}
?> 